import { badRequest, findSourcePackage, getOriginalCodeFrame, internalServerError, json, jsonString, noContent } from './shared';
import fs, { constants as FS } from 'fs/promises';
import path from 'path';
import { launchEditor } from '../internal/helpers/launchEditor';
import { getSourceMapFromFile } from '../internal/helpers/get-source-map-from-file';
const currentSourcesByFile = new Map();
export async function batchedTraceSource(project, frame) {
    const file = frame.file ? decodeURIComponent(frame.file) : undefined;
    if (!file) return;
    const sourceFrame = await project.traceSource(frame);
    if (!sourceFrame) return;
    let source = null;
    // Don't look up source for node_modules or internals. These can often be large bundled files.
    if (sourceFrame.file && !(sourceFrame.file.includes('node_modules') || sourceFrame.isInternal)) {
        let sourcePromise = currentSourcesByFile.get(sourceFrame.file);
        if (!sourcePromise) {
            sourcePromise = project.getSourceForAsset(sourceFrame.file);
            currentSourcesByFile.set(sourceFrame.file, sourcePromise);
            setTimeout(()=>{
                // Cache file reads for 100ms, as frames will often reference the same
                // files and can be large.
                currentSourcesByFile.delete(sourceFrame.file);
            }, 100);
        }
        source = await sourcePromise;
    }
    var _sourceFrame_line, _sourceFrame_column, _sourceFrame_methodName, _ref;
    return {
        frame: {
            file: sourceFrame.file,
            lineNumber: (_sourceFrame_line = sourceFrame.line) != null ? _sourceFrame_line : 0,
            column: (_sourceFrame_column = sourceFrame.column) != null ? _sourceFrame_column : 0,
            methodName: (_ref = (_sourceFrame_methodName = sourceFrame.methodName) != null ? _sourceFrame_methodName : frame.methodName) != null ? _ref : '<unknown>',
            arguments: []
        },
        source
    };
}
function createStackFrame(searchParams) {
    const fileParam = searchParams.get('file');
    if (!fileParam) {
        return undefined;
    }
    // rsc://React/Server/file://<filename>?42 => file://<filename>
    const file = fileParam.replace(/^rsc:\/\/React\/[^/]+\//, '').replace(/\?\d+$/, '');
    var _searchParams_get, _searchParams_get1, _searchParams_get2;
    return {
        file,
        methodName: (_searchParams_get = searchParams.get('methodName')) != null ? _searchParams_get : '<unknown>',
        line: parseInt((_searchParams_get1 = searchParams.get('lineNumber')) != null ? _searchParams_get1 : '0', 10) || 0,
        column: parseInt((_searchParams_get2 = searchParams.get('column')) != null ? _searchParams_get2 : '0', 10) || 0,
        isServer: searchParams.get('isServer') === 'true'
    };
}
export async function createOriginalStackFrame(project, frame) {
    const traced = await batchedTraceSource(project, frame);
    if (!traced) {
        const sourcePackage = findSourcePackage(frame);
        if (sourcePackage) return {
            sourcePackage
        };
        return null;
    }
    return {
        originalStackFrame: traced.frame,
        originalCodeFrame: getOriginalCodeFrame(traced.frame, traced.source),
        sourcePackage: findSourcePackage(traced.frame)
    };
}
export function getOverlayMiddleware(project) {
    return async function(req, res, next) {
        const { pathname, searchParams } = new URL(req.url, 'http://n');
        if (pathname === '/__nextjs_original-stack-frame') {
            const frame = createStackFrame(searchParams);
            if (!frame) return badRequest(res);
            let originalStackFrame;
            try {
                originalStackFrame = await createOriginalStackFrame(project, frame);
            } catch (e) {
                return internalServerError(res, e.message);
            }
            if (!originalStackFrame) {
                res.statusCode = 404;
                res.end('Unable to resolve sourcemap');
                return;
            }
            return json(res, originalStackFrame);
        } else if (pathname === '/__nextjs_launch-editor') {
            const frame = createStackFrame(searchParams);
            if (!frame) return badRequest(res);
            const fileExists = await fs.access(frame.file, FS.F_OK).then(()=>true, ()=>false);
            if (!fileExists) return noContent(res);
            try {
                var _frame_line, _frame_column;
                launchEditor(frame.file, (_frame_line = frame.line) != null ? _frame_line : 1, (_frame_column = frame.column) != null ? _frame_column : 1);
            } catch (err) {
                console.log('Failed to launch editor:', err);
                return internalServerError(res);
            }
            noContent(res);
        }
        return next();
    };
}
export function getSourceMapMiddleware(project, distDir) {
    return async function(req, res, next) {
        const { pathname, searchParams } = new URL(req.url, 'http://n');
        if (pathname !== '/__nextjs_source-map') {
            return next();
        }
        let filename = searchParams.get('filename');
        if (!filename) {
            return badRequest(res);
        }
        if (filename.startsWith('webpack://next/')) {
            return noContent(res);
        }
        try {
            if (filename.startsWith('/_next/static')) {
                filename = path.join(distDir, // /_next/static/chunks/%5Bproject%5D... => static/chunks/[project]...
                decodeURIComponent(filename.replace(/^\/_next\//, '')));
            }
            const sourceMapString = await project.getSourceMap(filename);
            if (sourceMapString) {
                return jsonString(res, sourceMapString);
            }
            if (filename.startsWith('file:') || filename.startsWith(path.sep)) {
                const sourceMap = await getSourceMapFromFile(filename);
                if (sourceMap) {
                    return json(res, sourceMap);
                }
            }
        } catch (error) {
            console.error('Failed to get source map:', error);
        }
        noContent(res);
    };
}

//# sourceMappingURL=middleware-turbopack.js.map